/* c4.c - "Connect 4"/"4 in a Row" style game.
 * (C) Paul A. Marshall 2002
 *
 * Requires front end user interface to be playable.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <stdlib.h>
#include "c4mac.h"
#include "strategy.h"
#include <time.h>

char board [COLS][ROWS];
int level [COLS];
int pdepth;
int movenum;
int lastmove;

/* carry out init necessary for new game */
void init_game(void)
{
	int i, j;
	
	movenum = 0;
	srand(time(0));
	for (i = 0; i < COLS; i++) {
		level[i] = 0;
		for (j = 0; j < ROWS; j++)
			board [i][j] = 0;
	}
	pwin_init();
	init_bitcount();
}
	
/* return the player who can force a win if this move is carried out or
 * return zero if inconclusive. This and the functions it calls are the only
 * parts that need to be efficient, the rest can be as bloated as you like ;-)
 */
char eval_move(int depth, char player, int move, int sdepth)
{
	int win(char player, int col);
	
	char result, opp_player;
	int win_count, poss_count, i;

	make_move(player, move);
	if (win(player, move)) {
		unmake_move(move);
		return player;
	}
	if (depth >= sdepth) {
		unmake_move(move);
		return 0;
	}
	opp_player = player ^ PLAYERFLIP;
	win_count = poss_count = 0;
	depth++;
	for (i = 0; i < COLS; i++) {
		if (level[i] < ROWS) {
			poss_count++;
			if ((result = eval_move(depth, opp_player, i, sdepth))) {
				if (result == opp_player) {
					unmake_move(move);
					return opp_player;
				}
				else
					win_count++;
			}
		}
	}
	unmake_move(move);
	if ((win_count == poss_count) && (poss_count != 0)) {
		return player;
	}
	return 0;
}

/* Sorry if this code looks like a dogs dinner. Readability has been sacrificed
 * for efficiency. Actually it's not that bad - it's just a series of unrolled
 * loops looking first one side of the most recently played piece, then the
 * other. (exception is vertical where all pieces will be below)
 */
/* return true if move indicated has resulted in win position */
int win(char player, int col)
{
	int row;

	row = level[col] - 1;
	
	/* check for vertical */
	if ((row > 2) && 
			(board[col][row - 1] == player) &&
			(board[col][row - 2] == player) &&
			(board[col][row - 3] == player))
			return 1;
	
	/* check for horizontal */
	if ((col > 0) &&
			(board[col - 1][row] == player)) {
		if ((col > 1) &&
				(board[col - 2][row] == player)) {
			if ((col > 2) &&
					(board[col - 3][row] == player))
				return 1;
			else
				if ((col < (COLS - 1)) &&
						(board[col + 1][row] == player))
					return 1;
		}
		else
			if ((col < (COLS - 2)) &&
					(board[col + 1][row] == player) &&
					(board[col + 2][row] == player))
				return 1;
	}
	else
		if ((col < (COLS - 3)) &&
				(board[col + 1][row] == player) &&
				(board[col + 2][row] == player) &&
				(board[col + 3][row] == player))
			return 1;
	
	/* check for diag / */
	if ((col > 0) && (row > 0) &&
			(board[col -1][row - 1] == player)) {
		if ((col > 1) && (row > 1) &&
				(board[col - 2][row - 2] == player)) {
			if ((col > 2) && (row > 2) &&
					(board[col -3][row - 3] == player))
				return 1;
			else
				if ((col < (COLS - 1)) && (row < (ROWS - 1)) &&
					(board[col + 1][row + 1] == player))
					return 1;
		}
		else
			if ((col < (COLS - 2)) && (row < (ROWS - 2)) &&
					(board[col + 1][row + 1] == player) &&
					(board[col + 2][row + 2] == player))
				return 1;
	}
	else
		if ((col < (COLS - 3)) && (row < (ROWS - 3)) &&
				(board[col + 1][row + 1] == player) &&
				(board[col + 2][row + 2] == player) &&
				(board[col + 3][row + 3] == player))
			return 1;
	
	/* check for diag \ */
	if ((col > 0) && (row < (ROWS - 1)) && 
			(board[col - 1][row + 1] == player)) {
		if ((col > 1) && (row < (ROWS - 2)) &&
				(board[col - 2][row + 2] == player)) {
			if ((col > 2) && (row < (ROWS - 3)) &&
					(board[col - 3][row + 3] == player))
				return 1;
			else
				if ((col < (COLS - 1)) && (row > 0) &&
					(board[col + 1][row - 1] == player))
					return 1;
		}
		else
			if ((col < (COLS - 2)) && (row > 1) &&
					(board[col + 1][row - 1] == player) &&
					(board[col + 2][row - 2] == player))
				return 1;
	}
	else
		if ((col < (COLS - 3)) && (row > 2) &&
				(board[col + 1][row - 1] == player) &&
				(board[col + 2][row - 2] == player) &&
				(board[col + 3][row - 3] == player))
			return 1;
	return 0;
}

/* choose a move for computer player */
int computer_move(char player)
{
	int choosegood(int move[], int c, char player, int depth);
	int choosebad(int move[], int c, char player, int depth);

	int badc, goodc, indc;
	int  badmove[COLS], goodmove[COLS], indmove[COLS];
	int i, move, depth;
	char result;
	
	badc = goodc = indc = 0;
	depth = pdepth + (movenum / COLS);
	for (i = 0; i < COLS; i++) {
		if (level[i] < ROWS) {
			if ((result = eval_move(1, player, i, depth))) {
				if (result == player) {
					goodmove[goodc++] = i;
				}
				else {
					badmove[badc++] = i;
				}
			}
			else {
				indmove[indc++] = i;
			}
		}
	}
	if (goodc > 0)
		move = choosegood(goodmove, goodc, player, depth);
	else 
		if (indc > 0)
			move = chooseind(indmove, indc, player);
		else 
			move = choosebad(badmove, badc, player, depth);
	make_move(player, move);
	lastmove = move;
	movenum++;
	if (win(player, move))
		return 1;
	else
		return 0;
}

/* Now that we know we are going to win, do it using the fewest moves */
int choosegood(int move[], int c, char player, int depth)
{
	int cmoves(int depth, char player, int move, int sdepth, int winner);
	
	int i, bestmove, tmp;
	int bestc = TOTALCELLS;

	for (i = 0; i < c; i++) {
		tmp = cmoves(1, player, move[i], depth, 1);
		if ((tmp < bestc) || ((tmp == bestc) && (rand() & 1))) {
			bestc = tmp;
			bestmove = move[i];
		}
	}
	return bestmove;
}

/* Now we know we can be forced to lose, prolong it for as long as possible.
 * With a bit of luck, we might throw the opponent off the scent
 */
int choosebad(int move[], int c, char player, int depth)
{
	int cmoves(int depth, char player, int move, int sdepth, int winner);

	int i, bestmove, tmp;
	int bestc = 0;

	for (i = 0; i < c; i++) {
		tmp = cmoves(1, player, move[i], depth, 0);
		if ((tmp > bestc) || ((tmp == bestc) && (rand() & 1))) {
			bestc = tmp;
			bestmove = move[i];
		}
	}
	return bestmove;
}
	
/* Figure out how many moves away a win is. The assumption is made that the
 * winning player will try to conclude the game in the fewest number of moves
 * whilst the losing player will drag it out for as long as possible.
 * The main search function stops looking once it knows that a particular
 * branch is a forced win which saves a lot of processing during fruitless
 * searches. This is why it has not been polluted with the task of collecting
 * this information.
 * This function is used only on moves that are known to have forced wins */
int cmoves(int depth, char player, int move, int sdepth, int winner)
{
	char opp_player;
	int result, rdepth, i;

	make_move(player, move);
	if (win(player,move)) {
		unmake_move(move);
		if (winner)
			return depth;
		else
			return TOTALCELLS;
	}
	if (depth >= sdepth) {
		unmake_move(move);
		return TOTALCELLS;
	}	
	opp_player = player ^ PLAYERFLIP;
	result = (winner ? 0 : TOTALCELLS);
	depth++;
	for (i = 0; i < COLS; i++) {
		if (level[i] < ROWS) {
			rdepth = cmoves(depth, opp_player, i, sdepth, !winner);
			if (winner)
				result = ((rdepth > result) ? rdepth : result);
			else
				result = ((rdepth < result) ? rdepth : result);
		}
	}
	unmake_move(move);
	return result;
}
	
/* record user move */
int user_move(char player, int move)
{
	make_move(player, move);
	lastmove = move;
	movenum++;
	return win(player, move);
}
