/* strategy.c */
/* This strategy has been inspired by the excellent work by James D. Allen
 * entitled "Expert Play in Connect-Four" although the implementation is not
 * nearly what it should be. With the hard work I have put in, I have managed
 * to get it nearly as good as random choosing ;-) ... a little more work is
 * required.
 * The job of these routines is to try to build up a strong board position from
 * a shortlist of moves vetted for combinatorial attacks */

#include "c4mac.h"
#include <stdio.h>
#include <stdlib.h>

/* We talk about TOTAL wins etc, but actually ignore vertical */
#define	TOTALWINS ((COLS - 3 ) * ROWS + (COLS - 3) * (ROWS - 3) * 2)
#define TOTALTHREATS ((TOTALWINS) * 4 * 2)

#define EVEN	0x02
#define ODD	0x01
#define MIXED	0x03
/* macro returns number of bits set in a given integer
 * (requires a correctly filled out lookup table
 */
#define bitcount(A) bitcountlu[(A)]

int bitcountlu[1 << COLS];
extern char board[COLS][ROWS];
extern int level[COLS];
struct pwin_struct {
	char	player;
	int	strength;
	char	type;
	int	col[4];
	int	row[4];
};
struct tlist_struct {
	struct pwin_struct *pwin;
	struct tlist_struct *next;
};

struct pwin_struct pwin[TOTALWINS];
struct tlist_struct tlist[TOTALTHREATS];
int tlisti;
struct tlist_struct p1tmap[COLS][ROWS];
struct tlist_struct p2tmap[COLS][ROWS];

struct tcount_struct {
	int odd;
	int even;
	int mixed;
};
struct plcount_struct {
	struct tcount_struct player1;
	struct tcount_struct player2;
	struct tcount_struct all;	/* may do away with all */
} pltcount[4];
struct colcount_struct {
	struct tcount_struct player1;
	struct tcount_struct player2;
	struct tcount_struct all;	/* may do away with all */
} colcount[4];

/* function to return bit position of lowest bit set in integer */
int bitpos(int num)
{
	int i = 0;

	while (num) {
		num >>= 1;
		i++;
	}
	return (i - 1);
}

/* these init functions that follow need to be run once only prior to 
 * calling the other functions
 */

/* fill in all the column/row addresses into our table of potential wins */
void pwin_init(void)
{
	void exwin(int col, int row, int coldir, int rowdir,
			struct pwin_struct *pwin);
	int i, j;
	int pwini = 0;
	
	/* horizontal wins */
	/* we deliberately store the top row horizontals first to facilitate
	 * checking for a particular type of threat (see p2toprow())
	 */
	for (j = (ROWS - 1); j >= 0; j--)
		for (i = 0; i < (COLS - 3); i++)
			exwin(i, j, 1, 0, &pwin[pwini++]);
	/* / diag wins */
	for (i = 0; i < (COLS - 3); i++)
		for (j = 0; j < (ROWS - 3); j++)
			exwin(i, j, 1, 1, &pwin[pwini++]);
	/* \ diag wins */
	for (i = 0; i < (COLS - 3); i++)
		for (j = 3; j < ROWS; j++)
			exwin(i, j, 1, -1, &pwin[pwini++]);
	/* check we have created the number of potential wins that we thought
	 * we would. Error message if we have managed to get this far without
	 * seg faulting. These lines can be removed once it checks out ok
	 */
	if (pwini != TOTALWINS) {
		fprintf(stderr, "\nIn pwin init, reckoned on %d, but got %d\n",
				TOTALWINS, pwini);
		exit(0);
	}
}

/* extract four column/row addresses on a given */
void exwin(int col, int row, int coldir, int rowdir, struct pwin_struct *pwin)
{
	int i;

	for (i = 0; i < 4; i++) {
		pwin->col[i] = col;
		pwin->row[i] = row;
		col += coldir;
		row += rowdir;
	}
}

/* init set bit count lookup table */
void init_bitcount(void)
{
	int i, j, k;

	for (i = 0; i < (1 << COLS); i++) {
		bitcountlu[i] = 0;
		k = i;
		for (j = 0; j < COLS; j++) {
			if (k & 1)
				bitcountlu[i]++;
			k = k >> 1;
		}
	}
}
			
	
/* put together threat maps for each player */
void build_maps(void)
{
	void assesswin(struct pwin_struct *pwin);
	
	int i, j;
	
	tlisti = 0;
	for (i = 0; i < COLS; i++)
		for (j = 0; j < ROWS; j++) {
			p1tmap[i][j].pwin = p2tmap[i][j].pwin = NULL;
			p1tmap[i][j].next = p2tmap[i][j].next = NULL;
		}
	for (i = 0; i < TOTALWINS; i++)
		assesswin(&pwin[i]);	
}

void assesswin(struct pwin_struct *pwin)
{
	void store_threats(char player, struct pwin_struct *pwin);
	
	int i;
	int p1c = 0;
	int p2c = 0;
	int emptyc = 0;
	int androw = 1;
	int orrow = 0;
	char cell;

	for (i = 0; i < 4; i++) {
		cell = board[pwin->col[i]][pwin->row[i]];
		if (cell == 0) {
			emptyc++;
			androw &= pwin->row[i];
			orrow |= pwin->row[i];
		}
		else {
			if (cell == PLAYER1)
				p1c++;
			else
				p2c++;
		}
	}
	orrow &= 1;
	/* we are dealing with indices. odd index means even row */
	if (androw)
		pwin->type = EVEN;
	else {
		if (orrow)
			pwin->type = MIXED;
		else
			pwin->type = ODD;
	}
	/* will do the "right thing" when 4 empties */
	pwin->player = 0;
	if ((p1c + emptyc) == 4) {
		pwin->strength = p1c;
		store_threats(PLAYER1, pwin);
	}
	if ((p2c + emptyc) == 4) {
		pwin->strength = p2c;
		store_threats(PLAYER2, pwin);
	}
}

/* make entries into player's threat map for given potential win */
void store_threats(char player, struct pwin_struct *pwin)
{
	void add_list(struct tlist_struct *threat, struct pwin_struct *pwin);
		
	int i;
	int row, col;

	pwin->player |= player;
	for (i = 0; i < 4; i++) {
		col = pwin->col[i];
		row = pwin->row[i];
		if (board[col][row] == 0) {
			if (player == PLAYER1)
				add_list(&p1tmap[col][row], pwin);
			else
				add_list(&p2tmap[col][row], pwin);
		}
	}
}

/* Add threat record to a given threat map list */
void add_list(struct tlist_struct *threatp, struct pwin_struct *pwin)
{
	struct tlist_struct *p;
	
	if (threatp->pwin == NULL)
		threatp->pwin = pwin;
	else {
		p = threatp;
		while (p->next != NULL)
			p = p->next;
		p->next = &tlist[tlisti];
		/* we want strength 3's to be at the top of the list */
		if (pwin->strength == 3) {
			tlist[tlisti].pwin = threatp->pwin;
			threatp->pwin = pwin;
		}
		else
			tlist[tlisti].pwin = pwin;
		tlist[tlisti].next = NULL;
		tlisti++;
		if (tlisti > TOTALTHREATS) {
			fprintf(stderr, "In add_list, threat list overflow");
			exit(0);
		}
	}
}

/* remove threats that can't be used */
void discount(char player)
{
	void discountlist(struct tlist_struct *threatp, char player);
	void discounttype(int col, int row, char player);
		
	int i, j;

	for (i = 0; i < COLS; i++)
		for (j = 0; j < ROWS; j++) {
			if (j == level[i]) {
				if (player == PLAYER1)
					discountlist(&p1tmap[i][j], PLAYER1);
				else
					discountlist(&p2tmap[i][j], PLAYER2);
			}
			if ((p1tmap[i][j].pwin != NULL) &&
					(p1tmap[i][j].pwin->strength == 3)) {
				discounttype(i, j + 1, PLAYER2);
				discountlist(p1tmap[i][j].next,PLAYER1);
			}
			if ((p2tmap[i][j].pwin != NULL) &&
					(p2tmap[i][j].pwin->strength == 3)) {
				discounttype(i, j + 1, PLAYER1);
				discountlist(p2tmap[i][j].next,PLAYER2);
				if (j & 1)
					discounttype(i, 1, PLAYER1);
			}
		}
}

/* discount threats of player on either odd rows or even in a particular col */
void discounttype(int col, int row, char player)
{
	void discountlist(struct tlist_struct *threatp, char player);
	
	while (row < ROWS) {
		if (player == PLAYER1)
			discountlist(&p1tmap[col][row], player);
		else
			discountlist(&p2tmap[col][row], player);
		row += 2;
	}
}

/* discount list of threats provided */
void discountlist(struct tlist_struct *threatp, char player)
{
	while (threatp != NULL) {
		if (threatp->pwin != NULL)
			threatp->pwin->player &= ~player;
		threatp = threatp->next;
	}
}

/* count the number of threats, organised into various categories */
void count_threats(void)
{
	int colflags(struct pwin_struct *pwin);
	void reset_counts(void);

	int i;

	reset_counts();
	for (i = 0; i < TOTALWINS; i++) {
		if (pwin[i].player & PLAYER1) {
			switch (pwin[i].type) {
				case ODD :
					pltcount[pwin[i].strength].player1.odd++;
					colcount[pwin[i].strength].player1.odd
						|= colflags(&pwin[i]);
					break;
				case EVEN :
					pltcount[pwin[i].strength].player1.even++;
					colcount[pwin[i].strength].player1.even
						|= colflags(&pwin[i]);
					break;
				case MIXED :
					pltcount[pwin[i].strength].player1.mixed++;
					colcount[pwin[i].strength].player1.mixed
						|= colflags(&pwin[i]);
			}
		}
		if (pwin[i].player & PLAYER2) {
			switch (pwin[i].type) {
				case ODD :
					pltcount[pwin[i].strength].player2.odd++;
					colcount[pwin[i].strength].player2.odd
						|= colflags(&pwin[i]);
					
					break;
				case EVEN :
					pltcount[pwin[i].strength].player2.even++;
					colcount[pwin[i].strength].player2.even
						|= colflags(&pwin[i]);
					break;
				case MIXED :
					pltcount[pwin[i].strength].player2.mixed++;
					colcount[pwin[i].strength].player2.mixed
						|= colflags(&pwin[i]);
			}
		}
	}
}

/* how lame is this? call the union! */
void reset_counts(void)
{
	int i;

	for (i = 0; i < 4; i++) {
		pltcount[i].player1.odd = 0;
		pltcount[i].player1.even = 0;
		pltcount[i].player1.mixed = 0;
		colcount[i].player1.odd = 0;
		colcount[i].player1.even = 0;
		colcount[i].player1.mixed = 0;
		pltcount[i].player2.odd = 0;
		pltcount[i].player2.even = 0;
		pltcount[i].player2.mixed = 0;
		colcount[i].player2.odd = 0;
		colcount[i].player2.even = 0;
		colcount[i].player2.mixed = 0;
		pltcount[i].all.odd = 0;
		pltcount[i].all.even = 0;
		pltcount[i].all.mixed = 0;
		colcount[i].all.odd = 0;
		colcount[i].all.even = 0;
		colcount[i].all.mixed = 0;
	}
}

/* return bitmap representing threat columns */
int colflags(struct pwin_struct *pwin)
{
	int i, col;
	int bitmap = 0;

	for (i = 0; i < 4; i++) {
		col = pwin->col[i];
		if (board[col][pwin->row[i]] == 0) {
			bitmap += 1 << col;
		}
	}
	return bitmap;
}

/* try to put a value to the current board position for the player concerned */
/* we calculate in player 1's favour and reverse it if necessary */
int evalboard(char player)
{
	int scorewin(void);
	int nextplay(struct pwin_struct *pwin);
	
	int i;
	int score, scadd;
	
	build_maps();
	discount(player);
	count_threats();
	score = scorewin();
	
	/* now we attempt to score none winning threats. This is a bit
	 * difficult to know how to do. This employs a "bird in the hand is
	 * worth two in the bush" system (that's a technical term)
	 */
	for (i = 0; i < TOTALWINS; i++) {
		if (pwin[i].player & PLAYER1) {
			if (pwin[i].type == ODD)
				scadd = 8 << pwin[i].strength;
			else
				scadd = 4 << pwin[i].strength;
			if ((player == PLAYER2) && nextplay(&pwin[i]))
				scadd = (scadd * 5) / 4;
			score += scadd;
		}
		if (pwin[i].player & PLAYER2) {
			if (pwin[i].type == EVEN)
				scadd = -(8 << pwin[i].strength);
			else
				scadd = -(4 << pwin[i].strength);
			if ((player == PLAYER1) && nextplay(&pwin[i]))
				scadd = (scadd * 5) / 4;
			score += scadd;
		}
	}
		
	if (player == PLAYER1)
		return score;
	else
		return -score;
}

/* search for threats which if left uncompromised, will win the game */
int scorewin(void)
{
	int pl1addmajor(void);
	int pl2addmajor(void);
	int p2toprow(void);
	
	int p1modd, p1cols;
	int p2modd, p2cols;
	int mcols;
	
	p1modd = pltcount[3].player1.odd;
	p1cols = colcount[3].player1.odd;
	if (pl1addmajor()) {
		p1modd++;
		p1cols |= colcount[2].player1.odd;
	}
	p2modd = pltcount[3].player2.odd;
	p2cols = colcount[3].player2.odd;
	if (pl2addmajor())
		p2modd++;
	mcols = bitcount(p1cols | p2cols);
			
	/* concentrate mostly on odd threats here */
	switch (p2modd) {
		case 0 :
			if (p1modd > 0)
				return 4096;
		case 1 :
			if (p1modd > 0)
				switch (mcols) {
					case 1 :
						return 4096;
					case 2 :
						if (p1cols == (p1cols | p2cols))
							return 4096;
						break;
					case 3 :
						return 4096;
					case 4 :
						return -4096;
				}
			break;
		default :
			if (p1modd == 0)
				return -4096;
			if (p1modd == 1)
				switch (mcols) {
					case 1 :
						return 4096;
					case 2 :
						if (p1cols == (p1cols | p2cols))
							return 4096;
						if (p2cols == (p1cols | p2cols))
							return -4096;
						break;
					case 3 :
						return 4096;
					case 4 :
						return -4096;
				}
			if (p1modd >= 2)
				switch (mcols) {
					case 2 :
						return -4096;
					case 3 :
						return 4096;
				}
	}

	/* if no score from odd threats, we look at player2 even threats */
	if ((pltcount[3].player2.even > 0) || (pltcount[2].player2.even > 0))
		return -4096;
	if (p2toprow())
		return -4096;
	
	return 0;
}

/* return 1 if one cell of supplied possible win is next row to be played.
 * Must be only one cell - the thinking being that only one cell can be
 * captured at a time and the rest of the threat can be easily refuted.
 */
int nextplay(struct pwin_struct *pwin)
{
	int i;
	int c = 0;

	for (i = 0; i < 4; i++)
		if (level[pwin->col[i]] == pwin->row[i])
			c++;
	if (c == 1)
		return 1;
	return 0;
}

/* do we have player1 minor threats that can be combined to be considered
 * a major threat?
 * 3 odd threat cells from strength 2 threats or..
 * 2 odd threat cells from strength 2 + even part of mixed or strength 3
 * in same col
 */
int pl1addmajor(void)
{
	int colsearch(int col, int startrow, char type, int strength, char player);
	
	int i;
	
	if (pltcount[2].player1.odd == 0)
		return 0;
	if (pltcount[2].player1.odd >= 2)
		return 1;	/* 2 threats means >= 3 threat cells */
	/* we know now that we have a single strength 2 odd threat */
	if (colcount[2].player1.odd & colcount[3].player1.even)
		return 1;	/* strength 3 even in same col */
	if ((i = (colcount[2].player1.odd & colcount[2].player1.mixed))) {
		if (bitcount(i) == 2)
			return 1; /* mixed threats cells in same cols as odd */
		if (colsearch(bitpos(i), 1, MIXED, 2, PLAYER1) != -1)
			return 1; /* even part of mixed in same col */
	}
	return 0;
}

/* Have we got a minor threat that we can upgrade? 
 * Mixed threat with even part in same col and lower than an odd strength 3
 */
int pl2addmajor(void)
{
	int colsearch(int col, int startrow, char type, int strength, char player);
	
	int i, j, k;

	i = (colcount[3].player1.odd | colcount[3].player2.odd) &
		colcount[2].player2.mixed;
	if (i == 0)
		return 0;
	while (bitcount(i) > 0) {
		j = bitpos(i);
		k = colsearch(j, 1, MIXED, 2, PLAYER2);
		if (k != -1) {
			if (colsearch(j, k + 1, ODD, 3, PLAYER1) != -1)
				return 1;
			if (colsearch(j, k + 1, ODD, 3, PLAYER2) != -1)
				return 1;
		}
		i ^= 1 << j;
	}
	return 0;
}

/* Search up a column for a given type of threat. Return row number where
 * threat occurs or -1.
 */
int colsearch(int col, int startrow, char type, int strength, char player)
{
	int i;
	struct tlist_struct *p;

	for (i = startrow; i < ROWS; i += 2) {
		if (player == PLAYER1)
			p = &p1tmap[col][i];
		else
			p = &p2tmap[col][i];
		while ((p != NULL) && (p->pwin != NULL)) {
			if ((p->pwin->type == type) && 
					(p->pwin->strength == strength))
				return i;
			p = p->next;
		}
	}
	return -1;
}

/* special, but very satisfying kind of threat - player 2 has piece on top row
 * bolstered from underneath (hell I can't describe it without a diagram :-( )
 */
int p2toprow(void)
{
	int i, j;

	for (i = 0; i < (COLS - 3); i++) {
		if ((pwin[i].player == PLAYER2) && (pwin[i].strength == 1)) {
			j = 0;
			while ((board[pwin[i].col[j]][ROWS - 1] == 0) &&
					(j < 4))
				j++;
			if (j == 4)
				return 0; /* this shouldn't happen */
			if (board[pwin[i].col[j]][ROWS - 2] == PLAYER2)
				return 1;
		}
	}
	return 0;
}

/* choose from shortlist of moves based on board evaluation */
int chooseind(int indmove[], int indc, char player)
{
	int bestweight = -999999;
	int bestmove;
	int cmpweight;
	int i;
	
	for (i = 0; i < indc; i++) {
		make_move(player, indmove[i]);
		cmpweight = evalboard(player);
		unmake_move(indmove[i]);
		if (cmpweight > bestweight) {
			bestmove = indmove[i];
			bestweight = cmpweight;
		}
		else {
			if ((cmpweight == bestweight) && (rand() & 1))
				bestmove = indmove[i];
		}
	}
	return bestmove;
}
