/* cursed.c - Cursed Invaders, a space invaders style game based on ncurses
 * Copyright (C) Paul A. Marshall  2001
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "cursed.h"
#include <sys/time.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

struct itimerval ticks;
volatile sig_atomic_t	cleaning = 0;
volatile sig_atomic_t	clocktick = 0;
int		invaderticks, invaderrow, invadercount;

char		motherflags;
int		motherx, mothertimer;

bombt		bomb[MAXBOMBS];

char		barricade [BARRICADE_H][BARRICADE_W];
bool		have_colour;

char 		direction;
invt		invader[INVADERCOLS][INVADERROWS];
invt		*invaderp;
char		landed;

int		shipx, lives, missilex, missiley, score;
char		goleft, goright, fire;

void tick_counter (int signum);
void init_invaders(int row);
void init_ship(void);
void waitfortick(int lasttick);

int main(int argc, char *argv[])
{
	int keypress;
	
	void init_curses(void);
	void init_ticker(void);
	void play_game(void);
	
	init_curses();
	init_ticker();
	srand(time(0));

	do {
		nodelay(stdscr, FALSE);
		move (5,0);
		addstr("\n\tWelcome to Cursed Invaders: Version ");
		addstr(VERSION);
		addstr("\n\n\tA *TERRIFYING* game of skill by Paul Marshall");
		addstr("\n\n\tPress 's' to begin game");
		addstr("\n\tPress 'q' to exit\n\n");
		refresh();
		do {
			keypress = tolower(getch());
		} while ((keypress != 's') && (keypress != 'q'));
		if (keypress == 's')
			play_game();
	} while (keypress != 'q');
	
	endwin();
	return(0);
}

/* set up ncurses the way we want it */
void init_curses(void)
{
	int y, x;

	void termresize(int signum);
	void terminationhandlers(void);
	
	initscr();
	terminationhandlers();
	getmaxyx(stdscr, y, x);
	if ((y < 24) || (x < 80)) {
		endwin();
		puts("This program requires a terminal with at least");
		puts("24 x 80 chars, sorry.");
		exit(1);
	}
	if (signal(SIGWINCH, termresize) == SIG_ERR) {
		endwin();
		perror("Setting up SIGWINCH");
		exit(1);
	}
	keypad(stdscr, TRUE);
	cbreak();
	noecho();
	nodelay(stdscr, TRUE);

	if ((have_colour = has_colors())) {
		start_color();
		init_pair(COLOR_GREEN, COLOR_GREEN, COLOR_BLACK);
		init_pair(COLOR_RED, COLOR_RED, COLOR_BLACK);
		init_pair(COLOR_CYAN, COLOR_CYAN, COLOR_BLACK);
		init_pair(COLOR_WHITE, COLOR_WHITE, COLOR_BLACK);
		init_pair(COLOR_MAGENTA, COLOR_MAGENTA, COLOR_BLACK);
		init_pair(COLOR_BLUE, COLOR_BLUE, COLOR_BLACK);
		init_pair(COLOR_YELLOW, COLOR_YELLOW, COLOR_BLACK);
		bkgdset(' ' | COLOR_PAIR(COLOR_WHITE));
		erase();
		
	}
}

/* signal handling to cleanup terminal prior to termination */
void terminationhandlers(void)
{
	int *p;
	
	void cleanup(int signum);
	
	p = siglist;
	while (*p) {
		signal(*p, cleanup);
		p++;
	}
}

void cleanup(int signum)
{
	if (cleaning)
		raise(signum);
	cleaning = 1;
	endwin(); /* returns ERR if already out of curses mode, we don't care */
	signal(signum, SIG_DFL);
	raise(signum);
}
	
/* signal handler for resize of terminal */
void termresize(int signum)
{
	int y, x;
	
	endwin();
	initscr();
	refresh();
	getmaxyx(stdscr, y, x);
	if ((y < 24) || (x < 80)) {
		endwin();
		puts("\nWindow resized to less than 24 x 80, I give in\n");
		raise(SIGTERM);
	}
}
		
/* main game loop */
void play_game(void)
{
	int lastclocktick;
	int shiptick, invadertick, missiletick, bombtick, mothertick;
	
	void move_ship(void);
	void check_keys(void);
	void move_missile(void);
	void process_invaders(void);
	void init_bombs(void);
	void move_bombs(void);
	void init_barricades(int row);
	void init_mother(void);
	void move_mother(void);

	erase();
	invaderrow = 2;
	init_invaders(invaderrow);
	init_mother();
	init_bombs();
	score = 0;
	mvprintw(0, 0, "Score: %06d", 0);
	lives = LIVES;
	init_ship();
	missiley = 0;
	
	refresh();

	invadertick = shiptick = missiletick = bombtick = mothertick = 1;
	
	while (!landed) {
		lastclocktick = clocktick;
		check_keys();
		
		if (--shiptick <= 0) {
			move_ship();
			shiptick = SHIP_TICKS;
		}
		if (--invadertick <= 0) {
			process_invaders();
			invadertick = invaderticks;
		}
		if (--missiletick <= 0) {
			move_missile();
			missiletick = MISSILE_TICKS;
		}
		if (--bombtick <= 0 ) {
			move_bombs();
			bombtick = BOMB_TICKS;
		}
		if (--mothertick <= 0) {
			move_mother();
			mothertick = MOTHER_TICKS;
		}
		/* Bug here, whereby invaders start so low that they have
		 * already landed. If you can make it happen, you should
		 * get out more :p
		 */
		if (invadercount == 0)
			init_invaders(++invaderrow);
		move(0, 78); /* how do i make the cursor disappear? */
		refresh();
		waitfortick(lastclocktick);
	}
}

/* time management functions */
void init_ticker(void)
{
	if (signal(SIGALRM, tick_counter) == SIG_ERR) {
		perror("Setting up SIGALRM");
		exit(1);
	}
	ticks.it_interval.tv_usec = TICK_INTERVAL;
	ticks.it_interval.tv_sec = 0;
	ticks.it_value.tv_usec = TICK_INTERVAL;
	ticks.it_value.tv_sec = 0;
	if (setitimer(ITIMER_REAL, &ticks, NULL)) {
		perror("Setting itimer");
		exit(1);
	}
}

void tick_counter(int signum)
{
	clocktick++;
}
		
/* this wait for signal code taken from libc manual */
void waitfortick(int lasttick)
{
	sigset_t sigmask, oldsigmask;

	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGALRM); 
	sigprocmask(SIG_BLOCK, &sigmask, &oldsigmask);
	while (clocktick == lasttick) {
		sigsuspend(&oldsigmask);
	}
		sigprocmask(SIG_UNBLOCK, &sigmask, NULL);
}
	
/* invader management functions */
void init_invaders(int row)
{
	int i, j;
	
	void paint_invader(chtype *p, chtype colour);
	void display_invader(invt *invader);
	void init_barricades(int row);
	
	if (have_colour) {
		paint_invader(invaderpic, COLOR_PAIR(COLOR_MAGENTA));
		paint_invader(invaderpic + 8, COLOR_PAIR(COLOR_GREEN));
		paint_invader(invaderpic + 16, COLOR_PAIR(COLOR_RED));
		paint_invader(invaderpic + 24, COLOR_PAIR(COLOR_CYAN));
	}
	
	direction = RIGHT;
	landed = 0;
	invaderticks = INVADER_TICKS;
	invadercount = INVADERTOTAL;

	for(i = 0; i < INVADERROWS; i++)
		for(j = 0; j < INVADERCOLS; j++) {
			if (i<4)
				invader[j][i].type = i;
			else
				invader[j][i].type = 3;
			invader[j][i].y = i*2+row;
			invader[j][i].x = j*4;
			invader[j][i].status = FRAME1;
			display_invader(&invader[j][i]);
		}
	init_barricades(row + 2 * INVADERROWS);
}

void paint_invader(chtype *p, chtype colour)
{
	int i;

	for (i = 0; i < 8; i++)
		*(p++) |= colour;
}

void display_invader(invt *invader)
{
	chtype *p;

	p = invaderpic + (invader->type << 3) + (invader->status & FRAME1);
	mvaddch(invader->y, invader->x, *p);
	addch(*(++p));
	addch(*(++p));
}

void process_invaders(void)
{
	
	void process_invader(invt *invader);
	
	if (direction & DOWN)
		direction = (direction & RIGHT) ^ RIGHT;
	if (direction & NEXTDOWN)
		direction = (direction ^ NEXTDOWN) | DOWN;
	
	for (invaderp = invader[0]; invaderp < invader[0] +
		INVADERTOTAL; invaderp++)
		process_invader(invaderp);
}
	
void process_invader(invt *invader)
{
	void move_invader(invt *invader);
	void display_invader(invt *invader);
	void drop_bomb(int y, int x);
	void barricade_plough(int y, int x);
	
	if (!(invader->status & DEAD)) {
		mvaddstr(invader->y, invader->x, "   ");
		if (invader->status & EXPLODING) {
			invader->status = DEAD;
			invaderticks = (--invadercount * INVADER_TICKS);
			invaderticks /= INVADERTOTAL;
			invaderticks++;
		}
		else {
			invader->status ^= FRAME1;
			move_invader(invader);
			/* this might bite in future as the NEXTDOWN bit will be
			 * set during the down movement too. Fow now, it is
			 * cleared after completion of a down movement */
			if (invader->x == 0 || invader->x == 76)
				direction |= NEXTDOWN;
			if (invader->y == SHIPY)
				landed = 1;
			display_invader(invader);
			barricade_plough(invader->y, invader->x);
			if ((rand() % invadercount) == 0)
				drop_bomb(invader->y + 1, invader->x + 1);
		}
	}
}

void move_invader(invt *invader)
{
	if (direction & DOWN)
		(invader->y)++;
	else
		if (direction & RIGHT)
			(invader->x)++;
		else
			(invader->x)--;
}

/* return true if x,y coords supplied are within an invader */
int hit_invader(int y, int x)
{
	invt *invaderp;	
	int xi;
	chtype explosion = EXPLOSION;
	
	void addscore(int add);

	for (invaderp = invader[0]; invaderp < invader[0] +
		INVADERTOTAL; invaderp++) {
		if (!(invaderp->status & DEAD) && invaderp->y == y) {
			xi = invaderp->x;
			if (x >= xi && x <= (xi + 2)) {
				invaderp->status = EXPLODING;
				if (have_colour)
					explosion |= COLOR_PAIR(COLOR_YELLOW);
				mvaddch(y, xi, explosion);
				addch(explosion);
				addch(explosion);
				addscore(50);
				return (1);
			}
		}
	}
	return (0);
}

/* bomb management functions */
void init_bombs(void)
{
	int i;

	for (i = 0; i < MAXBOMBS; i++)
		bomb[i].y = 0;	
}

void drop_bomb(int y, int x)
{
	int i;

	for (i = 0; i< MAXBOMBS; i++) {
		if (bomb[i].y == 0) {
			bomb[i].y = y;
			bomb[i].x = x;
			mvaddch(y, x, BOMB);
			break;
		}
	}
}

void move_bombs(void)
{
	int i;
	
	int hit_barricade(int y, int x);
	void lose_life(void);

	for (i = 0; i < MAXBOMBS; i++) {
		if (bomb[i].y != 0) {
			mvaddch(bomb[i].y++, bomb[i].x, ' ');
			if (bomb[i].y > SHIPY)
				bomb[i].y = 0;
			else
				mvaddch(bomb[i].y, bomb[i].x, BOMB);
			if (bomb[i].y == SHIPY && bomb[i].x >= shipx &&
				bomb[i].x <= (shipx + 4))
				lose_life();
			if (hit_barricade(bomb[i].y, bomb[i].x))
				bomb[i].y = 0;
		}
	}
}

/* ship management functions */
void init_ship(void)
{
	shipx = 10;
	mvaddstr(SHIPY, 0, "Press Return to begin new life.....");
	refresh();
	nodelay(stdscr, FALSE);
	while (getch() != '\n')
		;
	nodelay(stdscr, TRUE);
	mvaddstr(SHIPY, 0, "                                    ");
	mvprintw(0, 40, "Lives Left: %d", lives - 1);
	mvaddstr(SHIPY, shipx, SHIP);
	refresh();
}

void check_keys(void)
{
	int keypress;
	
	fire = goleft = goright = 0;
	do {
		keypress = getch();
		switch (keypress) {
		case KEY_LEFT:
			goleft++;
			break;
		case KEY_RIGHT:
			goright++;
			break;
		case ' ':
			fire++;
		}
	} while (keypress != ERR);
}

void move_ship(void)
{
	if (goleft && shipx > 0) {
		mvaddstr(SHIPY, shipx, "     ");
		mvaddstr(SHIPY, --shipx, SHIP);
	}
	if (goright && shipx < 73) {
		mvaddstr(SHIPY, shipx, "     ");
		mvaddstr(SHIPY, ++shipx, SHIP);
	}
}

void lose_life(void)
{
	/* to lose one life is unfortunate.... */

	int i;
	
	mvaddstr(SHIPY, shipx, "*****");
	refresh();
	for (i = 0; i < 15; i++)
		waitfortick(clocktick);
	mvaddstr(SHIPY, shipx, "     ");
	if ((--lives) > 0)
		init_ship();
	else
		landed = 1;
}

/* process ship's missile */
void move_missile(void)
{
	int hit_invader(int y, int x);
	int hit_barricade(int y, int x);
	int hit_mother(int y, int x);

	if (missiley) {
		mvaddch(missiley--, missilex, ' ');
		if (hit_invader(missiley, missilex) || 
			hit_barricade(missiley, missilex) ||
			hit_mother(missiley, missilex))
			missiley = 0;
		if (missiley)
			mvaddch(missiley, missilex, MISSILE);
	}
	else {
		if (fire) {
			missiley = SHIPY - 1;
			missilex = shipx + 2;
			mvaddch(missiley, missilex, MISSILE);
		}
	}
}

/* keep the scores on the doors */
void addscore(int add)
{
	score += add;
	mvprintw(0, 7, "%06d", score);
}

/* barricade management functions */
void init_barricades(row)	/* row = highest barricade row allowed */
{
	int i, j;
	chtype block;
	
	block = ' ' | A_REVERSE;
	if (have_colour)
		block |= COLOR_PAIR(COLOR_YELLOW);
	
	row -= BARRICADE_Y;
	
	for (i = 0; i < BARRICADE_H; i++) {
		if (i < row)
			memset(barricade[i], 0, sizeof(char) * BARRICADE_W);
		else
			if (i == (BARRICADE_H - 1))
				memcpy(barricade[i], b_lastrow, sizeof(char)
					* BARRICADE_W);
			else
				memcpy(barricade[i], b_row, sizeof(char) *
					BARRICADE_W);
		for (j = 0; j < BARRICADE_W; j++)
			if (barricade[i][j] == 1)
				mvaddch(i + BARRICADE_Y, j, block);
	}
}

/* return true if x,y coords supplied contain barricade */
int hit_barricade(int y, int x)
{
	if ((y < BARRICADE_Y) || (y >= BARRICADE_Y + BARRICADE_H))
		return (0);

	y -= BARRICADE_Y;

	if (barricade[y][x] == 0)
		return (0);

	mvaddch(y + BARRICADE_Y, x, ' ');
	barricade[y][x] = 0;
	return (1);
}

/* remove barricade block occupied by x,y coords supplied */
void barricade_plough(int y, int x)
{
	if ((y >= BARRICADE_Y) && (y < (BARRICADE_Y + BARRICADE_H))) {
		y -= BARRICADE_Y;
		barricade[y][x] = 0;
		barricade[y][x + 1] = 0;
		barricade[y][x + 2] = 0;
	}
}

/* mothership management functions */
void init_mother(void)
{
	int i;

	mothertimer = MOTHER_INTERVAL;
	motherflags = 0;
	if (have_colour)
		for (i = 0; i < 6; i++) {
			mother_r[i] |= COLOR_PAIR(COLOR_BLUE);
			mother_l[i] |= COLOR_PAIR(COLOR_BLUE);
		}
}

void move_mother(void)
{
	void display_mother(void);
	
	if (!(motherflags & MOTHER_ON)) {
		if (!(--mothertimer)) {
			motherflags |= MOTHER_ON;
			motherflags ^= MOTHER_RIGHT;
			if (motherflags & MOTHER_RIGHT)
				motherx = 0;
			else
				motherx = 73;
			display_mother();
		}
	}
	else {
		mvaddstr(MOTHER_Y, motherx, "      ");
		if (motherflags & MOTHER_RIGHT)
			motherx++;
		else
			motherx--;
		if ((motherx < 0) || (motherx > 73) ||
			(motherflags & MOTHER_EXPLODING)) {
			motherflags &= ~(MOTHER_ON | MOTHER_EXPLODING);
			mothertimer = MOTHER_INTERVAL;
		}
		else
			display_mother();
	}
}

void display_mother(void)
{
		
	chtype *p;
	int i;

	if (motherflags & MOTHER_RIGHT)
		p = mother_r;
	else
		p = mother_l;
	move(MOTHER_Y, motherx);
	for (i = 0; i < 6; i++)
		addch(*(p++));
}

/* return true if x,y coords supplied are within mothership */
int hit_mother(int y, int x)
{
	if ((y != MOTHER_Y) || !(motherflags & MOTHER_ON))
		return(0);
	if ((x < motherx) || (x > (motherx + 5)))
		return(0);
	mvaddstr(MOTHER_Y, motherx, "******");
	motherflags |= MOTHER_EXPLODING;
	addscore(MOTHER_SCORE);
	return(1);
}
